import React, { useState, useEffect } from "react";
import axios from "axios";
import Layout from "./components/Layout";
import Sidebar from "./components/Sidebar";
import RequestPanel from "./components/RequestPanel";
import ResponsePanel from "./components/ResponsePanel";

export default function App() {
  const [url, setUrl] = useState("http://localhost:4500/api/test");
  const [method, setMethod] = useState("GET");
  const [body, setBody] = useState("{\n  \n}");
  const [response, setResponse] = useState(null);
  const [history, setHistory] = useState([]);
  const [loading, setLoading] = useState(false);

  // Initial fetch
  useEffect(() => {
    fetchHistory();
  }, []);

  const sendRequest = async () => {
    setLoading(true);
    setResponse(null);
    const startTime = Date.now();

    try {
      let parsedBody = {};
      try {
        parsedBody = body ? JSON.parse(body) : {};
      } catch (e) {
        alert("Invalid JSON in body");
        setLoading(false);
        return;
      }

      const proxyRes = await axios.post("http://localhost:4500/api/test", {
        url,
        method,
        headers: {},
        body: parsedBody,
      });

      const endTime = Date.now();
      const duration = endTime - startTime;
      const responseSize = JSON.stringify(proxyRes.data.response).length;

      setResponse({
        status: proxyRes.data.status,
        body: proxyRes.data.response,
        time: duration,
        size: (responseSize / 1024).toFixed(2),
        headers: proxyRes.headers,
      });

      fetchHistory();
    } catch (err) {
      console.error(err);
      const endTime = Date.now();
      setResponse({
        status: err.response?.status || 500,
        body: err.response?.data || { error: err.message },
        time: endTime - startTime,
        size: 0,
        headers: {},
      });
    } finally {
      setLoading(false);
    }
  };

  const fetchHistory = async () => {
    try {
      const res = await axios.get("http://localhost:4500/api/test/history");
      setHistory(res.data);
    } catch (err) {
      console.error("Failed to fetch history", err);
    }
  };

  const loadRequest = (req) => {
    setUrl(req.url);
    setMethod(req.method);
    const bodyStr = req.body
      ? typeof req.body === "string"
        ? req.body
        : JSON.stringify(req.body, null, 2)
      : "{}";
    setBody(bodyStr);
  };

  const deleteHistory = async (id) => {
    try {
      await axios.delete(`http://localhost:4500/api/test/history/${id}`);
      fetchHistory();
    } catch (error) {
      console.log(error);
    }
  };

  return (
    <Layout
      sidebar={
        <Sidebar
          history={history}
          onSelectHistory={loadRequest}
          onFetchHistory={fetchHistory}
          onDeleteHistory={deleteHistory}
        />
      }
    >
      <div className="flex flex-col h-full">
        <div className="h-1/2 overflow-hidden">
          <RequestPanel
            url={url}
            method={method}
            body={body}
            setUrl={setUrl}
            setMethod={setMethod}
            setBody={setBody}
            onSend={sendRequest}
          />
        </div>
        <div className="h-1/2 overflow-hidden border-t border-gray-800">
          <ResponsePanel response={response} />
        </div>
      </div>
    </Layout>
  );
}
